package gov.va.med.mhv.rxrefill.service.impl;

import java.math.BigDecimal;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.domain.model.DtoListPayload;
import gov.va.med.domain.model.StatusPayload;
import gov.va.med.domain.service.messaging.ErrorResponse;
import gov.va.med.domain.service.messaging.IMessageOriginator;
import gov.va.med.domain.service.messaging.MessageExchangePatternType;
import gov.va.med.domain.service.messaging.impl.MessageOriginator;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.common.data.model.UserProfile;
import gov.va.med.mhv.rxrefill.data.model.Institution;
import gov.va.med.mhv.rxrefill.data.model.Prescription;
import gov.va.med.mhv.rxrefill.data.model.PrescriptionRequest;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.model.RequestAttempt;
import gov.va.med.mhv.rxrefill.enums.MailWindowEnumeration;
import gov.va.med.mhv.rxrefill.enums.PrescriptionStatusEnumeration;
import gov.va.med.mhv.rxrefill.enums.RequestFunctionEnumeration;
import gov.va.med.mhv.rxrefill.service.RxRequestVistaService;
import gov.va.med.mhv.rxrefill.service.impl.util.RxRequestMessages;
import gov.va.med.mhv.rxrefill.util.RequestUtils;

@Component
public class RxRequestVistaServiceImpl implements RxRequestVistaService {

	private static Logger log = LogManager.getLogger(RxRequestVistaServiceImpl.class);

	private static final int MAX_SIG_LENGTH = 1024;

	private static Map<String, String> domainStatusCodes = new HashMap<String, String>() {
		{
			put("ACTIVE", "A");
			put("NON-VERIFIED", "NV");
			put("HOLD", "H");
			put("SUSPENDED", "S");
			put("EXPIRED", "E");
			put("DELETED", "DEL");
			put("DISCONTINUED", "D");
			put("DISCONTINUED BY PROVIDER", "DBP");
			put("DISCONTINUED (EDIT)", "DE");
			put("PROVIDER HOLD", "PH");
			put("OK", "OK");
		}
	};

	private static HashMap<String, String> statusCodes = new HashMap<String, String>() {
		{
			put("A", "active");
			put("NV", "nonVerified");
			put("H", "hold");
			put("S", "suspended");
			put("E", "expired");
			put("DEL", "deleted");
			put("D", "discontinued");
			put("DBP", "discontinuedByProvider");
			put("DE", "discontinuedEdit");
			put("OK", "pending");
		}
	};

	private static HashMap<String, MailWindowEnumeration> mailWindowCodes = new HashMap<String, MailWindowEnumeration>() {
		{
			put("M", MailWindowEnumeration.MAIL);
			put("W", MailWindowEnumeration.WINDOW);
		}
	};

	static {
		final String resourceName = "domainContainerContext.xml";
		if (log.isDebugEnabled()) {
			try {
				URL url = MessageOriginator.class.getClassLoader().getResource(resourceName);
				log.info("URL(domainContainer)=" + url);
			} catch (Exception e) {
				log.error("URL(domainContainer)=?", e);
			}
			try {
				URL url = RxRequestVistaServiceImpl.class.getClassLoader().getResource(resourceName);
				log.info("URL(/domainContainer)=" + url);
			} catch (Exception e) {
				log.error("URL(/domainContainer)=?", e);
			}
		}
	}

	@Autowired
	private RequestUtils requestUtils;

	@Autowired
	private IMessageOriginator messageOriginator;

	/**
	 * Execute the SendRxRequest service
	 */
	@Override
	public ResponseUtil sendRxRequest(Patient patient, Request request) {
		if (log.isDebugEnabled()) {
			log.debug("inside sendRxRequest");
		}

		ResponseUtil response = new ResponseUtil();
		List<Prescription> prescriptions = new ArrayList<Prescription>();
		long startTime = 0;
		if (log.isDebugEnabled()) {
			startTime = System.currentTimeMillis();
		}

		try {
			if (log.isDebugEnabled()) {
				log.debug("after messageOriginator");
			}

			gov.va.med.domain.service.messaging.Request domainRequest = createRxRefillRequest(patient, request);
			gov.va.med.domain.service.messaging.Response domainResponse = messageOriginator.sendRxRefillRequest(domainRequest);

			if (log.isDebugEnabled()) {
				log.debug("domainResponse.isOk() " + domainResponse.isOk());
				log.debug("domainResponse.getPayload().getClass().toString() " + domainResponse.getPayload().getClass().toString());
			}

			if (domainResponse != null && domainResponse.isOk() && (domainResponse.getPayload() instanceof DtoListPayload)) {
				gov.va.med.domain.model.DtoListPayload prescriptionsDtoList = (gov.va.med.domain.model.DtoListPayload) domainResponse.getPayload();

				//TODO: verify list parsing - SRE
				for (gov.va.med.domain.model.Prescription domainPrescription : ((List<gov.va.med.domain.model.Prescription>)prescriptionsDtoList.getDtoList())) {
					Prescription prescription = (requestUtils.isRefillRequest(request)) ? transferRefillPrescription(domainPrescription) : transferProfilePrescription(domainPrescription);
					prescription.setUserId(patient.getUserProfile().getId());
					prescriptions.add(prescription);
				}
				response.setSuccess(true);
				response.setPojoObject(prescriptions);
			} else if (domainResponse != null && domainResponse.getPayload() instanceof StatusPayload) {
				addMessage(response, patient, request, RxRequestMessages.STATUS_ERROR, RxRequestMessages.createStatusErrorMessage(request), describe((StatusPayload) domainResponse.getPayload()));
			} else {
				addMessage(response, patient, request, RxRequestMessages.MESSAGING_ERROR, RxRequestMessages.createMessagingErrorMessage(request), describe((ErrorResponse) domainResponse));
			}
		} catch (Exception e) {
			log.error("Error in sendRxRequest() " + e);
			// this is needed to prevent transaction rollback by interceptor
			addMessage(response, patient, request, RxRequestMessages.UNEXPECTED_ERROR, RxRequestMessages.createUnexpectedErrorMessage(request), e.getMessage()+ "\n\r" + ExceptionUtils.getStackTrace(e));
		}

		if (log.isDebugEnabled()) {
			long stopTime = System.currentTimeMillis();
			long time = stopTime - startTime;
			log.debug("PERFORMANCE - Vista RX Request took " + time + "ms");
		}

		return response;
	}

	private String describe(StatusPayload statusPayload) {
		return new StringBuilder()
			.append("Status=")
			.append(statusPayload.getStatus())
		    .append(";StatusDescription=")
		    .append(statusPayload.getStatusDescription())
		    .toString();
	}

	private String describe(ErrorResponse errorResponse) {
		return new StringBuilder()
			.append("ErrorCode=")
			.append(errorResponse.getStatusCode())
		    .append(";ErrorCause=")
		    .append(errorResponse.getCause())
		    .toString();
	}

	private void addMessage(ResponseUtil response, Patient patient, Request request, String key, String message, String description) {
		log.error(new StringBuilder().
			append("VistA messaging error occurred for").
			append(" PatientID=").
			append((patient != null) ? patient.getId() : "?").
			append(";RequestID=").
			append((request != null) ? request.getId() : "?").
			append(";").
			append(description).
			toString());
		response.getInfoMessages().put(key, message);
	}

	private String stripSsnFormatting(String ssn) {
		if (ssn == null || ssn.trim().length() == 0) {
			return ssn;
		}
		StringBuffer buff = new StringBuffer(ssn);
		for (int i = buff.length() - 1; i >= 0; i--) {
			if (!Character.isDigit(buff.charAt(i))) {
				buff.deleteCharAt(i);
			}
		}
		return buff.toString();
	}

	private gov.va.med.domain.model.Patient constructDomainPatient(Patient patient) {
		UserProfile userProfile = patient.getUserProfile();
		String ssn = userProfile.getSsn();
		gov.va.med.domain.model.Patient domainPatient = new gov.va.med.domain.model.Patient();
		((gov.va.med.domain.model.User) domainPatient).setSocialSecurityNumber(stripSsnFormatting(ssn));
		domainPatient.setIcn(patient.getIcn());
		domainPatient.setId(userProfile.getId());
		return domainPatient;
	}

	private gov.va.med.domain.service.messaging.Request createRxRefillRequest(Patient patient, Request request) {
		if (log.isDebugEnabled()) {
			log.debug("inside createRxRefillRequest. getRequestFunction::"+request.getRequestFunction()+"   ---   Request Id::"+request.getId());
		}

		// Patient
		gov.va.med.domain.model.Patient domainPatient = constructDomainPatient(patient);
		// MessageType
		gov.va.med.domain.service.messaging.MessageType messageType = constructMessageType(request.getRequestFunction());
		// destination
		gov.va.med.domain.model.Institution domainInstitution = new gov.va.med.domain.model.Institution();
		domainInstitution.setStationNumber(request.getInstitution().getStationNumber());
		domainInstitution.setId(request.getInstitution().getInstitutionId());
		// dates
		Date fromDate = request.getStartDate();
		Date toDate = request.getEndDate();
		// payload
		long mcid = request.getId().longValue();
		// to prevent duplicate mcids, use attempt id instead of request id as
		// mcid
		Set<RequestAttempt> attempts = request.getRequestAttempts();
		if (null != attempts && attempts.size() > 0) {
			mcid = 0;
			for(RequestAttempt attempt : request.getRequestAttempts()) {
				if (log.isDebugEnabled()) {
					log.debug("inside for loop - No of Attempts::"+attempts.size() + "Attempt Id is::"+attempt.getId());
				}
				if (attempt.getId().longValue() > mcid) {
					mcid = attempt.getId().longValue();
				}
			}
		}

		gov.va.med.domain.model.RxRefillPayload payload = new gov.va.med.domain.model.RxRefillPayload(new Long(mcid), (Long) null, domainPatient, domainInstitution, fromDate, toDate);
		// request
		gov.va.med.domain.service.messaging.Request domainRequest = new gov.va.med.domain.service.messaging.Request(messageType, payload);

		Set<PrescriptionRequest> prescriptionRequests = request.getPrescriptionRequests();

		if (null != prescriptionRequests) {
			if (log.isDebugEnabled()) {
				log.debug("prescriptionRequests Count::"+prescriptionRequests.size());
			}
		}

		if (null != prescriptionRequests && prescriptionRequests.size() > 0) {
			List<gov.va.med.domain.model.Prescription> domainPrescriptionsList = new ArrayList<gov.va.med.domain.model.Prescription>();
			for (PrescriptionRequest prescriptionRequest : prescriptionRequests) {

				if (log.isDebugEnabled()) {
					log.debug("prescriptionRequest getPrescriptionNumber::"+prescriptionRequest.getPrescription().getPrescriptionNumber());
				}

				gov.va.med.domain.model.Prescription domainPrescription = new gov.va.med.domain.model.Prescription();
				domainPrescription.setPrescriptionNum(prescriptionRequest.getPrescription().getPrescriptionNumber());
				domainPrescriptionsList.add(domainPrescription);
			}
			gov.va.med.domain.model.Prescription[] domainPrescriptions = domainPrescriptionsList.toArray(new gov.va.med.domain.model.Prescription[0]);
			payload.setPrescriptionList(domainPrescriptions);
		}
		return domainRequest;
	}

	private gov.va.med.domain.service.messaging.MessageType constructMessageType(int requestFunction) {
		gov.va.med.domain.service.messaging.MessageType messageType = null;
		switch (requestFunction) {
		case 1:
			messageType = new gov.va.med.domain.service.messaging.MessageType(RequestFunctionEnumeration.GETICN.getDescription(),
				MessageExchangePatternType.SYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);
			break;
		case 2:
			messageType = new gov.va.med.domain.service.messaging.MessageType(RequestFunctionEnumeration.REFILLPRESCRIPTION.getDescription(),
				MessageExchangePatternType.SYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);
			break;
		case 3:
			messageType = new gov.va.med.domain.service.messaging.MessageType(RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS.getDescription(),
				MessageExchangePatternType.SYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);
			break;
		case 4:
			messageType = new gov.va.med.domain.service.messaging.MessageType(RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getDescription(),
				MessageExchangePatternType.SYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);
			break;
		default:
			messageType = new gov.va.med.domain.service.messaging.MessageType(RequestFunctionEnumeration.GETPRESCRIPTIONUPDATE.getDescription(),
				MessageExchangePatternType.ASYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);
		}
		return messageType;
	}

	private Prescription transferProfilePrescription(gov.va.med.domain.model.Prescription domainPrescription) {
		Prescription prescription = new Prescription();
		prescription.setDaysSupply(domainPrescription.getDaysSupply());
		prescription.setDivision(new Long(domainPrescription.getDivision().longValue()));
		prescription.setDivisionName(domainPrescription.getDivisionName());
		prescription.setDrugName(domainPrescription.getDrugName());
		prescription.setExpirationCancelDate(domainPrescription.getExpirationOrCancelDate());
		prescription.setIen(domainPrescription.getIen());
		Institution institution = new Institution();
		institution.setInstitutionId(domainPrescription.getInstitutionId());
		prescription.setInstitution(institution);
		prescription.setIssueDateTime(domainPrescription.getIssueDate());
		prescription.setLastFillDate(domainPrescription.getLastFillDate());
		prescription.setMailWindow(mailWindowCodes.get(domainPrescription.getMailOrWindow()).getDescription());
		String phamacyStatus = domainPrescription.getPharmacyStatus();
		String statusCode = domainStatusCodes.get(phamacyStatus);
		statusCode = statusCodes.get(statusCode);

		if (log.isDebugEnabled()) {
			log.debug(String.format("phamacyStatus : %s statusCode : %s", phamacyStatus, statusCode));
		}

		prescription.setStatus(PrescriptionStatusEnumeration.fromValue(statusCode).getValue());
		prescription.setMhvStatus(domainPrescription.getMhvStatus());

		if (log.isDebugEnabled()) {
			log.debug(String.format("prescription.status : %s prescription.mhvStatus : %s", prescription.getStatus(), prescription.getMhvStatus()));
		}

		prescription.setMhvStatusDate(domainPrescription.getMhvStatusDate());
		prescription.setNumberOfRefills(domainPrescription.getNumReFill());
		prescription.setPlacerOrderNumber(domainPrescription.getPlaceOrderNum());
		prescription.setProviderFirstName(domainPrescription.getProviderFirstName());
		prescription.setProviderLastName(domainPrescription.getProviderLastname());
		prescription.setProviderId(domainPrescription.getProviderId());
		prescription.setQuantity(new BigDecimal(domainPrescription.getQuantity().longValue()));
		prescription.setPrescriptionNumber(domainPrescription.getPrescriptionNum());
		prescription.setReleaseDateTime(domainPrescription.getReleaseDate());
		prescription.setRemarks(domainPrescription.getRemarks());
		prescription.setSig(StringUtils.substring(domainPrescription.getSig(), 0, MAX_SIG_LENGTH));

		return prescription;
	}

	private Prescription transferRefillPrescription(gov.va.med.domain.model.Prescription domainPrescription) {
		Prescription prescription = new Prescription();
		Institution institution = new Institution();
		institution.setInstitutionId(domainPrescription.getInstitutionId());
		prescription.setInstitution(institution);
		prescription.setPrescriptionNumber(domainPrescription.getPrescriptionNum());

		if (log.isDebugEnabled()) {
			log.debug("domainPrescription.getMhvStatus() : " + domainPrescription.getMhvStatus());
		}

		int mhvStatus = (domainPrescription.getMhvStatus() != null) ? domainPrescription.getMhvStatus().intValue() : 0;
		if ((mhvStatus == 1) || (mhvStatus == -2)) {
			// filed or pending - then submitted
			prescription.setStatus(PrescriptionStatusEnumeration.SUBMITTED.getValue());
		}

		prescription.setMhvStatus(domainPrescription.getMhvStatus());

		return prescription;
	}
}
